"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDataSourceConnectionsRoutes = registerDataSourceConnectionsRoutes;
var _configSchema = require("@osd/config-schema");
var _common = require("../../../common");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerDataSourceConnectionsRoutes(router, defaultClient) {
  router.get({
    path: `${_common.API.DATA_SOURCE.CONNECTIONS}/{id?}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const client = request.params.id ? context.dataSource.opensearch.legacy.getClient(request.params.id).callAPI : defaultClient.asScoped(request).callAsCurrentUser;
    try {
      const resp = await client('enhancements.getDataConnections', {});
      return response.ok({
        body: resp
      });
    } catch (error) {
      if (error.statusCode === 404 || error.statusCode === 400) {
        return response.ok({
          body: []
        });
      }
      // Transform 500 errors to 503 to indicate service availability issues
      const statusCode = error.statusCode === 500 ? 503 : error.statusCode || 503;
      return response.custom({
        statusCode,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_common.API.DATA_SOURCE.ASYNC_JOBS}`,
    validate: {
      query: _configSchema.schema.object({
        id: _configSchema.schema.string(),
        queryId: _configSchema.schema.nullable(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const client = request.query.id ? context.dataSource.opensearch.legacy.getClient(request.query.id).callAPI : defaultClient.asScoped(request).callAsCurrentUser;
      const resp = await client('enhancements.getJobStatus', {
        queryId: request.query.queryId
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      // Transform 500 errors to 503 to indicate service availability issues
      const statusCode = error.statusCode === 500 ? 503 : error.statusCode || 503;
      return response.custom({
        statusCode,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_common.API.DATA_SOURCE.ASYNC_JOBS}`,
    validate: {
      query: _configSchema.schema.object({
        id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        query: _configSchema.schema.string(),
        datasource: _configSchema.schema.string(),
        lang: _configSchema.schema.string(),
        sessionId: _configSchema.schema.nullable(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const client = request.query.id ? context.dataSource.opensearch.legacy.getClient(request.query.id).callAPI : defaultClient.asScoped(request).callAsCurrentUser;
      const resp = await client('enhancements.runDirectQuery', {
        body: request.body
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      // Transform 500 errors to 503 to indicate service availability issues
      const statusCode = error.statusCode === 500 ? 503 : error.statusCode || 503;
      return response.custom({
        statusCode,
        body: error.message
      });
    }
  });
  router.delete({
    path: _common.API.DATA_SOURCE.ASYNC_JOBS,
    validate: {
      query: _configSchema.schema.object({
        id: _configSchema.schema.string(),
        queryId: _configSchema.schema.nullable(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const client = request.query.id ? context.dataSource.opensearch.legacy.getClient(request.query.id).callAPI : defaultClient.asScoped(request).callAsCurrentUser;
      await client('enhancements.deleteJob', {
        queryId: request.query.queryId
      });
      return response.noContent();
    } catch (error) {
      const statusCode = error.statusCode === 500 ? 503 : error.statusCode || 503;
      return response.custom({
        statusCode,
        body: error.message
      });
    }
  });
  /**
   * @experimental this API is experimental and might change in future releases
   */
  router.get({
    path: _common.API.DATA_SOURCE.REMOTE_CLUSTER.LIST,
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const client = request.query.dataSourceId ? await context.dataSource.opensearch.getClient(request.query.dataSourceId) : context.core.opensearch.client.asCurrentUser;
    try {
      const result = await client.transport.request({
        method: 'GET',
        path: '/_remote/info'
      }, {
        requestTimeout: 5000 // Enforce timeout to avoid hanging requests
      });

      const remoteClusters = Object.entries(result.body).map(([key, value]) => ({
        connectionAlias: key,
        ...value
      }));
      return response.ok({
        body: remoteClusters
      });
    } catch (error) {
      var _error$meta;
      const errMessage = (error === null || error === void 0 || (_error$meta = error.meta) === null || _error$meta === void 0 || (_error$meta = _error$meta.body) === null || _error$meta === void 0 ? void 0 : _error$meta.Message) || 'Unknown Error';
      // Transform 401 errors to 405 to indicate Method not allowed and remote cluster is not enabled
      return response.custom({
        statusCode: error.statusCode === 401 ? 405 : error.statusCode,
        body: {
          message: errMessage
        }
      });
    }
  });

  /**
   * @experimental this API is experimental and might change in future releases
   */
  router.get({
    path: _common.API.DATA_SOURCE.REMOTE_CLUSTER.INDEXES,
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string()),
        connectionAlias: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const client = request.query.dataSourceId ? await context.dataSource.opensearch.getClient(request.query.dataSourceId) : context.core.opensearch.client.asCurrentUser; // obtain the opensearch client corresponding to the datasourceId

    try {
      const result = await client.transport.request({
        method: 'GET',
        path: `/_resolve/index/${request.query.connectionAlias}:*`
      }, {
        requestTimeout: 5000 // Enforce timeout to avoid hanging requests
      });
      // @ts-expect-error TS7006 TODO(ts-error): fixme
      const indexes = result.body.indices.map(index => index.name);
      return response.ok({
        body: indexes
      });
    } catch (error) {
      var _error$meta2;
      const errMessage = (error === null || error === void 0 || (_error$meta2 = error.meta) === null || _error$meta2 === void 0 || (_error$meta2 = _error$meta2.body) === null || _error$meta2 === void 0 ? void 0 : _error$meta2.Message) || 'Unknown Error';
      return response.custom({
        statusCode: error.statusCode,
        body: {
          message: errMessage
        }
      });
    }
  });
}